/*
     (C) 2006 Peter Gruber

     This is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     You should have received a copy of the GNU General Public License
     along with libdoodle; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
 */


/* libdoodle_python.c
   ---------------------

   Implements the Python wrapper for libdoodle.
*/

/* Includes. */

#include <Python.h>
#include <doodle/doodle.h>
#include <string.h>
#include <structmember.h>

#define DEFAULTDBPATH "/var/lib/doodle/doodle.db"

#define ST_READWRITE 1

/* Typedefs. */

typedef struct
{
  PyObject_HEAD struct DOODLE_SuffixTree *suffixtree;
  char filename[1024];
  long count;
  unsigned long flags;
  PyObject *logger;
} SuffixTree;

typedef struct
{
  PyObject_HEAD SuffixTree *suffixtree;
  int pos;
} SuffixTreeIter;

typedef struct
{
  PyObject *list;
  int max_num;
} myfillret;

/* Type objects. */

static PyTypeObject SuffixTreeType;
static PyTypeObject SuffixTreeIterType;

void
mylog (void *context, unsigned int level, const char *message, ...)
{
  PyObject *arglist, *result;
  va_list ap;
  char tbuf[1024];

  va_start(ap, message);
  if (!context)
    return;
  if (!((SuffixTree *) context)->logger)
    return;
  if (!(PyCallable_Check (((SuffixTree *) context)->logger)))
    return;
  vsnprintf(tbuf, 1024, message, ap);
  va_end(ap);
  arglist = Py_BuildValue ("(ls)", level, tbuf);
  result = PyEval_CallObject (((SuffixTree *) context)->logger, arglist);
  Py_DECREF (arglist);
  if (result)
    {
      Py_DECREF (result);
    }
}

void
myfill (const DOODLE_FileInfo * fileinfo, void *arg)
{
  static char old[1024];
  int fl, ll;
  myfillret *r;
  PyObject *l, *n;

  if (!arg)
    return;
  r = (myfillret *) arg;
  l = r->list;
  fl = strlen (fileinfo->filename);
  ll = PyList_Size (l);
  if (r->max_num <= 0)
    return;
  if ((ll == 0) || (strncmp (old, fileinfo->filename, fl) != 0))
    {
      n = PyString_FromFormat ("%s", fileinfo->filename);
      PyList_Append (l, n);
      Py_DECREF (n);
      strncpy (old, fileinfo->filename, fl);
      (r->max_num)--;
    }
  return;
}

/* SuffixTree type. */

static PyObject *
SuffixTree_new (PyTypeObject * type, PyObject * args, PyObject * kwargs)
{
  SuffixTree *self;

  self = (SuffixTree *) type->tp_alloc (type, 0);
  if (self != NULL)
    self->suffixtree = NULL;
  return (PyObject *) self;
}

static int
SuffixTree_init (SuffixTree * self, PyObject * args, PyObject * kwargs)
{
  char *kwlist[] = { "dbname", "flags", "logger", NULL };
  PyObject *logger = NULL;
  char *fn;
  struct DOODLE_SuffixTree *tree;

  fn = DEFAULTDBPATH;
  if (!PyArg_ParseTupleAndKeywords
      (args, kwargs, "|skO", kwlist, &fn, &(self->flags), &logger))
    goto error;
  strncpy(self->filename, fn, 1024);
  if (self->flags & ST_READWRITE)
    {
      if (!(tree = DOODLE_tree_create (&mylog, self, self->filename)))
	goto error;
    }
  else
    {
      if (!(tree = DOODLE_tree_open_RDONLY (&mylog, self, self->filename)))
	goto error;
    }
  if (self->suffixtree)
    DOODLE_tree_destroy (self->suffixtree);
  self->suffixtree = tree;
  self->count = DOODLE_getFileCount (self->suffixtree);
  if (!logger)
    logger = Py_None;
  Py_INCREF (logger);
  self->logger = logger;
  return 0;
error:
  return -1;
}

static PyObject *
SuffixTree_str (SuffixTree * self)
{
  return PyString_FromFormat ("doodle.SuffixTree('%s')", self->filename);
}

static PyObject *
SuffixTree_limit (SuffixTree * self, PyObject * arg)
{
  PyObject *ret;

  if (!PyInt_Check (arg))
    return NULL;
  if (self->suffixtree)
    DOODLE_tree_set_memory_limit (self->suffixtree, PyInt_AsLong (arg));
  ret = Py_None;
  Py_INCREF (ret);
  return ret;
}

static PyObject *
SuffixTree_search (SuffixTree * self, PyObject * args)
{
  myfillret fret;
  PyObject *ret;
  char *ss;
  int mn = 100;

  if (!(self->suffixtree))
    return NULL;
  ret = Py_None;
  if (!PyArg_ParseTuple (args, "s|i", &ss, &mn))
    {
      Py_INCREF (ret);
    }
  else
    {
      fret.list = (ret = PyList_New (0));
      fret.max_num = mn;
      DOODLE_tree_search (self->suffixtree, ss, &myfill, &fret);
    }
  return ret;
}

static PyObject *
SuffixTree_close (SuffixTree * self, PyObject * args)
{
  PyObject *ret;

  if ((self->suffixtree))
    {
      DOODLE_tree_destroy (self->suffixtree);
      self->suffixtree = NULL;
      self->count = 0;
      self->filename[0] = '\0';
    }
  ret = Py_None;
  Py_INCREF (ret);
  return ret;
}

static PyObject *
SuffixTree_sync (SuffixTree * self, PyObject * args)
{
  PyObject *ret;

  if ((self->suffixtree) && (self->flags & ST_READWRITE))
    {
      DOODLE_tree_destroy (self->suffixtree);
      self->suffixtree = DOODLE_tree_create (&mylog, self, self->filename);
      self->count = DOODLE_getFileCount (self->suffixtree);
    }
  ret = Py_None;
  Py_INCREF (ret);
  return ret;
}

static PyObject *
SuffixTree_expand (SuffixTree * self, PyObject * args)
{
  char *ss, *fn;
  int ret;

  if ((!(self->suffixtree)) && (!(self->flags & ST_READWRITE)))
    return NULL;
  if (!PyArg_ParseTuple (args, "ss", &ss, &fn))
    ret = -1;
  else
    ret = DOODLE_tree_expand (self->suffixtree, ss, fn);
  self->count = DOODLE_getFileCount (self->suffixtree);
  return PyInt_FromLong (ret);
}

static PyObject *
SuffixTree_truncatedel (SuffixTree * self, PyObject * args)
{
  PyObject *ret;

  if (!(self->suffixtree))
    return NULL;
  DOODLE_tree_truncate_deleted (self->suffixtree, &mylog, self);
  ret = Py_None;
  Py_INCREF (ret);
  return ret;
}

static PyObject *
SuffixTree_truncatemod (SuffixTree * self, PyObject * args)
{
  PyObject *ret;

  if (!(self->suffixtree))
    return NULL;
  DOODLE_tree_truncate_modified (self->suffixtree, &mylog, self);
  ret = Py_None;
  Py_INCREF (ret);
  return ret;
}

static PyObject *
SuffixTree_dump (SuffixTree * self, PyObject * args)
{
  PyObject *fo;
  int ret;

  if (!(self->suffixtree))
    return NULL;
  ret = -1;
  if (PyArg_ParseTuple (args, "O", &fo))
    {
      if (PyFile_Check (fo))
	ret = DOODLE_tree_dump (PyFile_AsFile (fo), self->suffixtree);
    }
  return PyInt_FromLong (ret);
}

static PyObject *
SuffixTree_truncate (SuffixTree * self, PyObject * arg)
{
  int ret;

  if (!(self->flags & ST_READWRITE))
    return NULL;
  if (!PyString_Check (arg))
    ret = -1;
  else
    {
      Py_INCREF (arg);
      ret = DOODLE_tree_truncate (self->suffixtree, PyString_AsString (arg));
      Py_DECREF (arg);
    }
  self->count = DOODLE_getFileCount (self->suffixtree);
  return PyInt_FromLong (ret);
}

static PyObject *
SuffixTree_get (SuffixTree * self, PyObject * args)
{
  PyObject *ret;
  DOODLE_FileInfo *fi;
  long idx = 0;

  if (!(self->suffixtree))
    return NULL;
  if (PyInt_Check (args))
    idx = PyInt_AsLong (args);
  if (idx < 0 || idx >= self->count)
    idx = 0;
  fi =
    (DOODLE_FileInfo *) DOODLE_getFileAt (self->suffixtree,
					  (unsigned int) idx);
  ret = PyTuple_New (2);
  PyTuple_SetItem (ret, 0, PyString_FromFormat ("%s", fi->filename));
  PyTuple_SetItem (ret, 1, PyInt_FromLong (fi->mod_time));
  return ret;
}

static PyObject *
SuffixTree_searchApprox (SuffixTree * self, PyObject * args,
			 PyObject * kwargs)
{
  myfillret fret;
  PyObject *ret;
  char *ss;
  int ap = 0, ig = 0, mn = 100;
  char *kwlist[] = { "ss", "approx", "ignorecase", "maxresults", NULL };

  if (!(self->suffixtree))
    return NULL;
  ret = Py_None;
  if (!PyArg_ParseTupleAndKeywords
      (args, kwargs, "s|iii", kwlist, &ss, &ap, &ig, &mn))
    {
      Py_INCREF (ret);
    }
  else
    {
      fret.list = (ret = PyList_New (0));
      fret.max_num = mn;
      DOODLE_tree_search_approx (self->suffixtree, ap, ig, ss, &myfill,
				 &fret);
    }
  return ret;
}

static PyObject *
SuffixTree_iter (SuffixTree * self)
{
  PyObject *ret;

  ret = (PyObject *) PyObject_New (SuffixTreeIter, &SuffixTreeIterType);
  Py_INCREF (self);
  ((SuffixTreeIter *) ret)->suffixtree = self;
  ((SuffixTreeIter *) ret)->pos = 0;
  return ret;
}

static void
SuffixTree_dealloc (SuffixTree * self)
{
  if (self->suffixtree)
    DOODLE_tree_destroy (self->suffixtree);
  self->ob_type->tp_free ((PyObject *) self);
}

static PyMethodDef SuffixTree_methods[] = {
  {"setMemoryLimit", (PyCFunction) SuffixTree_limit, METH_O},
  {"sync", (PyCFunction) SuffixTree_sync, METH_NOARGS},
  {"close", (PyCFunction) SuffixTree_close, METH_NOARGS},
  {"expand", (PyCFunction) SuffixTree_expand, METH_VARARGS},
  {"truncate", (PyCFunction) SuffixTree_truncate, METH_O},
  {"truncateDeleted", (PyCFunction) SuffixTree_truncatedel, METH_NOARGS},
  {"truncateModified", (PyCFunction) SuffixTree_truncatemod, METH_NOARGS},
  {"getFileAt", (PyCFunction) SuffixTree_get, METH_O},
  {"dump", (PyCFunction) SuffixTree_dump, METH_O},
  {"treeSearch", (PyCFunction) SuffixTree_search, METH_VARARGS},
  {"treeSearchApprox", (PyCFunction) SuffixTree_searchApprox,
   METH_VARARGS | METH_KEYWORDS},
  {NULL}			/* Sentinel */
};

static PyMemberDef SuffixTree_members[] = {
  {"filename", T_STRING_INPLACE, offsetof (SuffixTree, filename), READONLY,
   "database filename"},
  {"count", T_LONG, offsetof (SuffixTree, count), READONLY,
   "number of indexed files"},
  {"flags", T_ULONG, offsetof (SuffixTree, flags), READONLY,
   "flags"},
  {"logger", T_OBJECT, offsetof (SuffixTree, logger), 0,
   "Logging Callback"},
  {NULL}			/* Sentinel */
};

static PyTypeObject SuffixTreeType = {
  PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
  "doodle.SuffixTree",		/*tp_name */
  sizeof (SuffixTree),		/*tp_basicsize */
  0,				/*tp_itemsize */
  (destructor) SuffixTree_dealloc,	/*tp_dealloc */
  0,				/*tp_print */
  0,				/*tp_getattr */
  0,				/*tp_setattr */
  0,				/*tp_compare */
  0,				/*tp_repr */
  0,				/*tp_as_number */
  0,				/*tp_as_sequence */
  0,				/*tp_as_mapping */
  0,				/*tp_hash */
  0,				/*tp_call */
  (reprfunc) SuffixTree_str,	/*tp_str */
  0,				/*tp_getattro */
  0,				/*tp_setattro */
  0,				/*tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/*tp_flags */
  "SuffixTree object",		/* tp_doc */
  0,				/* tp_traverse */
  0,				/* tp_clear */
  0,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  (getiterfunc) SuffixTree_iter,	/* tp_iter */
  0,				/* tp_iternext */
  SuffixTree_methods,		/* tp_methods */
  SuffixTree_members,		/* tp_members */
  0,				/* tp_getset */
  0,				/* tp_base */
  0,				/* tp_dict */
  0,				/* tp_descr_get */
  0,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  (initproc) SuffixTree_init,	/* tp_init */
  0,				/* tp_alloc */
  SuffixTree_new,		/* tp_new */
};

static PyObject *
SuffixTreeIter_new (PyTypeObject * type, PyObject * args, PyObject * kwargs)
{
  SuffixTreeIter *self;

  self = (SuffixTreeIter *) type->tp_alloc (type, 0);
  if (self != NULL)
    {
      self->suffixtree = NULL;
      self->pos = 0;
    }
  return (PyObject *) self;
}

static int
SuffixTreeIter_init (SuffixTreeIter * self, PyObject * args,
		     PyObject * kwargs)
{
  if (self->suffixtree)
    {
      Py_DECREF (self->suffixtree);
      self->suffixtree = NULL;
    }
  self->pos = 0;
  return 0;
}

static PyObject *
SuffixTreeIter_iter (SuffixTreeIter * self)
{
  Py_INCREF (self);
  return (PyObject *) self;
}

static PyObject *
SuffixTreeIter_iternext (SuffixTreeIter * self)
{
  PyObject *ret;
  DOODLE_FileInfo *fi;

  if (!(self->suffixtree))
    return NULL;
  if (!(self->suffixtree->suffixtree))
    return NULL;
  if (self->pos >= self->suffixtree->count)
    return NULL;
  fi =
    (DOODLE_FileInfo *) DOODLE_getFileAt (self->suffixtree->suffixtree,
					  (unsigned int) self->pos);
  ret = PyTuple_New (3);
  PyTuple_SetItem (ret, 0, PyInt_FromLong (self->pos));
  PyTuple_SetItem (ret, 1, PyString_FromFormat ("%s", fi->filename));
  PyTuple_SetItem (ret, 2, PyInt_FromLong (fi->mod_time));
  (self->pos)++;
  return ret;
}

static void
SuffixTreeIter_dealloc (SuffixTreeIter * self)
{
  if (self->suffixtree)
    {
      Py_DECREF (self->suffixtree);
    }
  self->ob_type->tp_free ((PyObject *) self);
}

static PyTypeObject SuffixTreeIterType = {
  PyObject_HEAD_INIT (NULL) 0,	/*ob_size */
  "doodle.SuffixTree.SuffixTreeIter",	/*tp_name */
  sizeof (SuffixTreeIter),	/*tp_basicsize */
  0,				/*tp_itemsize */
  (destructor) SuffixTreeIter_dealloc,	/*tp_dealloc */
  0,				/*tp_print */
  0,				/*tp_getattr */
  0,				/*tp_setattr */
  0,				/*tp_compare */
  0,				/*tp_repr */
  0,				/*tp_as_number */
  0,				/*tp_as_sequence */
  0,				/*tp_as_mapping */
  0,				/*tp_hash */
  0,				/*tp_call */
  0,				/*tp_str */
  0,				/*tp_getattro */
  0,				/*tp_setattro */
  0,				/*tp_as_buffer */
  Py_TPFLAGS_DEFAULT,		/*tp_flags */
  "Iterator for indexed files",	/* tp_doc */
  0,				/* tp_traverse */
  0,				/* tp_clear */
  0,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  (getiterfunc) SuffixTreeIter_iter,	/* tp_iter */
  (iternextfunc) SuffixTreeIter_iternext,	/* tp_iternext */
  0,				/* tp_methods */
  0,				/* tp_members */
  0,				/* tp_getset */
  0,				/* tp_base */
  0,				/* tp_dict */
  0,				/* tp_descr_get */
  0,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  (initproc) SuffixTreeIter_init,	/* tp_init */
  0,				/* tp_alloc */
  SuffixTreeIter_new,		/* tp_new */
};

/* Module level. */

static PyObject *
Doodle_defaultdb (PyObject * self)
{
  PyObject *ret;

  ret = PyString_FromFormat ("%s", DEFAULTDBPATH);
  return ret;
}

static PyMethodDef Extractor_Module_methods[] = {
  {"defaultdb", (PyCFunction) Doodle_defaultdb, METH_NOARGS},
  {NULL}			/* Sentinel */
};

PyMODINIT_FUNC
initdoodle (void)
{
  PyObject *m;

  if (PyType_Ready (&SuffixTreeType))
    return;
  if (PyType_Ready (&SuffixTreeIterType))
    return;

  if ((m =
       Py_InitModule3 ("doodle", Extractor_Module_methods,
		       "Doodle module.")) == NULL)
    return;
  Py_INCREF (&SuffixTreeType);
  PyModule_AddObject (m, "SuffixTree", (PyObject *) & SuffixTreeType);
  PyModule_AddIntConstant (m, "ST_READWRITE", ST_READWRITE);
}
