/*
     (C) 2006 Peter Gruber

     This is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     You should have received a copy of the GNU General Public License
     along with libextractor; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
 */


/* libextractor_python.c
   ---------------------

   Implements the Python wrapper for libextractor.
*/

/* Includes. */

#include <Python.h>
#include <structmember.h>
#include <extractor.h>
#include <string.h>
#include <stdlib.h>

/* Typedefs. */

typedef struct
{
  PyObject_HEAD EXTRACTOR_ExtractorList *extractors;
  PyObject *extractorlist;
} ExtractorList;

typedef struct
{
  PyObject_HEAD EXTRACTOR_KeywordList *keywords;
} KeywordList;

typedef struct
{
  PyObject_HEAD KeywordList *list;
  unsigned long pos;
} KeywordListIter;

/* Type objects. */

static PyTypeObject ExtractorListType;
static PyTypeObject KeywordListType;
static PyTypeObject KeywordListIterType;

/* ExtractorList type. */

PyObject *
getextractors (EXTRACTOR_ExtractorList * elist)
{
  PyObject *ret;
  EXTRACTOR_ExtractorList *nlist;
  int i;

  for (i = 0, nlist = elist; nlist; nlist = nlist->next)
    i++;
  ret = PyTuple_New (i);
  for (i = 0, nlist = elist; nlist; nlist = nlist->next, i++)
    {
      if (elist->options)
	PyTuple_SetItem (ret, i,
			 PyString_FromFormat ("%s(%s)", nlist->libname,
					      nlist->options));
      else
	PyTuple_SetItem (ret, i, PyString_FromString (nlist->libname));
    }
  return ret;
}

static PyObject *
ExtractorList_new (PyTypeObject * type, PyObject * args, PyObject * kwargs)
{
  ExtractorList *self;

  self = (ExtractorList *) type->tp_alloc (type, 0);
  if (self != NULL)
    self->extractors = NULL;
  return (PyObject *) self;
}

static int
ExtractorList_init (ExtractorList * self, PyObject * args, PyObject * kwargs)
{
  PyObject *mod = NULL, *iter = NULL, *item = NULL;
  EXTRACTOR_ExtractorList *elist;
  char *kwlist[] = { "modules", NULL };

  if (!PyArg_ParseTupleAndKeywords (args, kwargs, "|O", kwlist, &mod))
    goto error;
  elist = NULL;
  if (!mod || mod == Py_None)
    elist = EXTRACTOR_loadDefaultLibraries ();
  else if (PyString_Check (mod))
    elist = EXTRACTOR_loadConfigLibraries (NULL, PyString_AsString (mod));
  else if (PyList_Check (mod))
    {
      iter = PyObject_GetIter (mod);
      while ((item = PyIter_Next (iter)))
	{
	  if (!PyString_Check (item))
	    {
	      Py_DECREF (item);
	      goto error;
	    }
	  elist =
	    EXTRACTOR_loadConfigLibraries (elist, PyString_AsString (item));
	  Py_DECREF (item);
	}
    }
  else if (PyObject_IsInstance (mod, (PyObject *) & ExtractorListType))
    {
      mod = getextractors (((ExtractorList *) mod)->extractors);
      iter = PyObject_GetIter (mod);
      while ((item = PyIter_Next (iter)))
	{
	  elist =
	    EXTRACTOR_loadConfigLibraries (elist, PyString_AsString (item));
	  Py_DECREF (item);
	}
      Py_DECREF (mod);
    }
  else
    goto error;
  if (!elist)
    goto error;
  if (self->extractors)
    EXTRACTOR_removeAll (self->extractors);
  self->extractors = elist;
  self->extractorlist = getextractors (elist);
  return 0;
error:
  return -1;
}

static PyObject *
ExtractorList_del (ExtractorList * self, PyObject * arg)
{
  PyObject *iter = NULL, *item = NULL, *ret;
  EXTRACTOR_ExtractorList *elist, *nlist;

  ret = Py_None;
  Py_INCREF (ret);
  elist = self->extractors;
  if (PyString_Check (arg))
    nlist = EXTRACTOR_removeLibrary (elist, PyString_AsString (arg));
  else if (PyList_Check (arg))
    {
      iter = PyObject_GetIter (arg);
      nlist = elist;
      while ((item = PyIter_Next (iter)))
	{
	  if (PyString_Check (item))
	    nlist = EXTRACTOR_removeLibrary (nlist, PyString_AsString (item));
	  Py_DECREF (item);
	}
    }
  else
    return ret;
  self->extractors = nlist;
  self->extractorlist = getextractors (nlist);
  return ret;
}

static PyObject *
ExtractorList_addlast (ExtractorList * self, PyObject * arg)
{
  PyObject *iter = NULL, *item = NULL, *ret;
  EXTRACTOR_ExtractorList *elist, *nlist;

  ret = Py_None;
  Py_INCREF (ret);
  elist = self->extractors;
  if (PyString_Check (arg))
    nlist = EXTRACTOR_addLibraryLast (elist, PyString_AsString (arg));
  else if (PyList_Check (arg))
    {
      iter = PyObject_GetIter (arg);
      nlist = elist;
      while ((item = PyIter_Next (iter)))
	{
	  if (PyString_Check (item))
	    nlist =
	      EXTRACTOR_addLibraryLast (nlist, PyString_AsString (item));
	  Py_DECREF (item);
	}
    }
  else
    return ret;
  self->extractors = nlist;
  self->extractorlist = getextractors (nlist);
  return ret;
}

static PyObject *
ExtractorList_add (ExtractorList * self, PyObject * arg)
{
  PyObject *iter = NULL, *item = NULL, *ret;
  EXTRACTOR_ExtractorList *elist, *nlist;

  ret = Py_None;
  Py_INCREF (ret);
  elist = self->extractors;
  if (PyString_Check (arg))
    nlist = EXTRACTOR_addLibrary (elist, PyString_AsString (arg));
  else if (PyList_Check (arg))
    {
      iter = PyObject_GetIter (arg);
      nlist = elist;
      while ((item = PyIter_Next (iter)))
	{
	  if (PyString_Check (item))
	    nlist = EXTRACTOR_addLibrary (nlist, PyString_AsString (item));
	  Py_DECREF (item);
	}
    }
  else
    return ret;
  self->extractors = nlist;
  self->extractorlist = getextractors (nlist);
  return ret;
}

static PyObject *
ExtractorList_extractfromfile (ExtractorList * self, PyObject * arg)
{
  EXTRACTOR_KeywordList *kw;
  PyObject *ret;

  ret = Py_None;
  if (!PyString_Check (arg))
    {
      Py_INCREF (ret);
    }
  else if (!(kw =
	     EXTRACTOR_getKeywords (self->extractors,
				    PyString_AsString (arg))))
    {
      Py_INCREF (ret);
    }
  else
    {
      ret = (PyObject *) PyObject_New (KeywordList, &KeywordListType);
      ((KeywordList *) ret)->keywords = kw;
    }
  return ret;
}

static PyObject *
ExtractorList_extract (ExtractorList * self, PyObject * arg)
{
  EXTRACTOR_KeywordList *kw;
  PyObject *ret;

  ret = Py_None;
  if (!PyString_Check (arg))
    {
      Py_INCREF (ret);
    }
  else if (!(kw =
	     EXTRACTOR_getKeywords2 (self->extractors,
				     PyString_AsString (arg),
				     PyString_Size (arg))))
    {
      Py_INCREF (ret);
    }
  else
    {
      ret = (PyObject *) PyObject_New (KeywordList, &KeywordListType);
      ((KeywordList *) ret)->keywords = kw;
    }
  return ret;
}

static PyObject *
ExtractorList_str (ExtractorList * self)
{
  PyObject *ret, *list;
  int i;

  ret = PyString_FromString ("extractor.ExtractorList('");
  list = self->extractorlist;
  for (i = 0; i < PyTuple_Size (list); i++)
    {
      if (i > 0)
	PyString_ConcatAndDel (&ret, PyString_FromString ("', '"));
      PyString_Concat (&ret, PyTuple_GetItem (list, i));
    }
  PyString_ConcatAndDel (&ret, PyString_FromString ("')"));
  return ret;
}

static void
ExtractorList_dealloc (ExtractorList * self)
{
  if (self->extractors)
    {
      EXTRACTOR_removeAll (self->extractors);
      self->extractors = NULL;
    }
  self->ob_type->tp_free ((PyObject *) self);
}

static PyMethodDef ExtractorList_methods[] = {
  {"addExtractor", (PyCFunction) ExtractorList_add, METH_O},
  {"addExtractorLast", (PyCFunction) ExtractorList_addlast, METH_O},
  {"removeExtractor", (PyCFunction) ExtractorList_del, METH_O},
  {"extractFromFile", (PyCFunction) ExtractorList_extractfromfile, METH_O},
  {"extractFromString", (PyCFunction) ExtractorList_extract, METH_O},
  {NULL}			/* Sentinel */
};

static PyMemberDef ExtractorList_members[] = {
  {"extractors", T_OBJECT, offsetof (ExtractorList, extractorlist), READONLY,
   "List of Extractors"},
  {NULL}			/* Sentinel */
};

static PyTypeObject ExtractorListType = {
  PyObject_HEAD_INIT (NULL) 0,	/* ob_size */
  "extractor.ExtractorList",	/* tp_name */
  sizeof (ExtractorList),	/* tp_basicsize */
  0,				/* tp_itemsize */
  (destructor) ExtractorList_dealloc,	/* tp_dealloc */
  0,				/* tp_print */
  0,				/* tp_getattr */
  0,				/* tp_setattr */
  0,				/* tp_compare */
  0,				/* tp_repr */
  0,				/* tp_as_number */
  0,				/* tp_as_sequence */
  0,				/* tp_as_mapping */
  0,				/* tp_hash */
  0,				/* tp_call */
  (reprfunc) ExtractorList_str,	/* tp_str */
  0,				/* tp_getattro */
  0,				/* tp_setattro */
  0,				/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/* tp_flags */
  "Extractor List Objects",	/* tp_doc */
  0,				/* tp_traverse */
  0,				/* tp_clear */
  0,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  0,				/* tp_iter */
  0,				/* tp_iternext */
  ExtractorList_methods,	/* tp_methods */
  ExtractorList_members,	/* tp_members */
  0,				/* tp_getset */
  0,				/* tp_base */
  0,				/* tp_dict */
  0,				/* tp_descr_get */
  0,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  (initproc) ExtractorList_init,	/* tp_init */
  0,				/* tp_alloc */
  ExtractorList_new,		/* tp_new */
};

static PyObject *
KeywordList_new (PyTypeObject * type, PyObject * args, PyObject * kwargs)
{
  KeywordList *self;

  self = (KeywordList *) type->tp_alloc (type, 0);
  if (self != NULL)
    self->keywords = NULL;
  return (PyObject *) self;
}

static int
KeywordList_init (KeywordList * self, PyObject * args, PyObject * kwargs)
{
  PyObject *oldkeywordlist = NULL;
  EXTRACTOR_KeywordList *elist, *nlist;
  char *kwlist[] = { "keywordlist", NULL };

  if (!PyArg_ParseTupleAndKeywords
      (args, kwargs, "|O", kwlist, &oldkeywordlist))
    return -1;
  if (self->keywords)
    EXTRACTOR_freeKeywords (self->keywords);
  elist = NULL;
  nlist = NULL;
  self->keywords = nlist;
  if (oldkeywordlist
      && PyObject_IsInstance (oldkeywordlist, (PyObject *) & KeywordListType))
    {
      elist = ((KeywordList *) oldkeywordlist)->keywords;
      nlist = malloc (sizeof (EXTRACTOR_KeywordList));
      self->keywords = nlist;
      memcpy (nlist, elist, sizeof (EXTRACTOR_KeywordList));
      nlist->keyword = malloc (sizeof (char) * (strlen (elist->keyword) + 1));
      strcpy (nlist->keyword, elist->keyword);
      while (elist->next)
	{
	  nlist->next = malloc (sizeof (EXTRACTOR_KeywordList));
	  nlist = nlist->next;
	  elist = elist->next;
	  memcpy (nlist, elist, sizeof (EXTRACTOR_KeywordList));
	  nlist->keyword =
	    malloc (sizeof (char) * (strlen (elist->keyword) + 1));
	  strcpy (nlist->keyword, elist->keyword);
	}
    }
  return 0;
}

static PyObject *
KeywordList_str (KeywordList * self)
{
  PyObject *ret;
  int i = 0;
  EXTRACTOR_KeywordList *elist;

  ret = PyString_FromString ("extractor.KeywordList(");
  elist = self->keywords;
  do
    {
      PyString_ConcatAndDel (&ret,
			     (EXTRACTOR_isBinaryType (elist->keywordType)) ?
			     PyString_FromFormat ("(%d,<binary>)",
						  elist->keywordType) :
			     PyString_FromFormat ("(%d,'%s')",
						  elist->keywordType,
						  elist->keyword));
      if (elist->next)
	PyString_ConcatAndDel (&ret, PyString_FromString (","));
      i++;
    }
  while ((elist = elist->next));
  PyString_ConcatAndDel (&ret, PyString_FromString (")"));
  return ret;
}

static PyObject *
KeywordList_removedups (KeywordList * self, PyObject * arg)
{
  int flag = 0;
  PyObject *ret;

  ret = Py_None;
  Py_INCREF (ret);
  if (!PyArg_ParseTuple(arg, "|i", &flag))
    return ret;
  if (self->keywords)
    self->keywords = EXTRACTOR_removeDuplicateKeywords (self->keywords, flag);
  return ret;
}

static PyObject *
KeywordList_removeempty (KeywordList * self, PyObject * arg)
{
  PyObject *ret;

  ret = Py_None;
  Py_INCREF (ret);
  if (self->keywords)
    self->keywords = EXTRACTOR_removeEmptyKeywords (self->keywords);
  return ret;
}

static PyObject *
KeywordList_removetype (KeywordList * self, PyObject * arg)
{
  int type = 0;
  PyObject *ret;

  ret = Py_None;
  Py_INCREF (ret);
  if (PyInt_Check (arg))
    type = PyInt_AsLong (arg);
  if (self->keywords)
    self->keywords = EXTRACTOR_removeKeywordsOfType (self->keywords, type);
  return ret;
}

static PyObject *
KeywordList_iter (KeywordList * self)
{
  PyObject *ret;

  ret = (PyObject *) PyObject_New (KeywordListIter, &KeywordListIterType);
  Py_INCREF (self);
  ((KeywordListIter *) ret)->list = self;
  ((KeywordListIter *) ret)->pos = 0;
  return ret;
}

static PyObject *
KeywordList_asList (KeywordList * self, PyObject * arg)
{
  PyObject *ret;
  EXTRACTOR_KeywordList *elist;
  long type = -1;

  ret = PyList_New (0);
  if (!(elist = self->keywords))
    return NULL;
  if (!PyArg_ParseTuple (arg, "|k", &type))
    return NULL;
  do
    {
      if ((type == -1) || (elist->keywordType == type))
	PyList_Append (ret, PyString_FromString (elist->keyword));
    }
  while ((elist = elist->next));
  return ret;
}

static PyObject *
KeywordList_asDict (KeywordList * self, PyObject * arg)
{
  PyObject *ret, *p, *q, *r;
  EXTRACTOR_KeywordList *elist;

  ret = PyDict_New ();
  if (!(elist = self->keywords))
    return ret;
  while (elist)
    {
      p =
	PyString_FromString (EXTRACTOR_getKeywordTypeAsString
			     (elist->keywordType));
      q = PyString_FromString (elist->keyword);
      if ((r = PyDict_GetItem (ret, p)) != NULL)
	{
	  PyList_Append (r, q);
	  Py_DECREF (p);
	  Py_DECREF (q);
	}
      else
	{
	  r = PyList_New (1);
	  PyList_SetItem (r, 0, q);
	  PyDict_SetItem (ret, p, r);
	  Py_DECREF (p);
	  Py_DECREF (r);
	}
      elist = elist->next;
    }
  return ret;
}

static void
KeywordList_dealloc (KeywordList * self)
{
  if (self->keywords)
    EXTRACTOR_freeKeywords (self->keywords);
  self->ob_type->tp_free ((PyObject *) self);
}

static PyMethodDef KeywordList_methods[] = {
  {"removeDuplicates", (PyCFunction) KeywordList_removedups,
   METH_VARARGS},
  {"removeEmpty", (PyCFunction) KeywordList_removeempty,
   METH_NOARGS},
  {"removeType", (PyCFunction) KeywordList_removetype, METH_O},
  {"as_list", (PyCFunction) KeywordList_asList, METH_VARARGS},
  {"as_dict", (PyCFunction) KeywordList_asDict, METH_NOARGS},
  {NULL}			/* Sentinel */
};

static PyTypeObject KeywordListType = {
  PyObject_HEAD_INIT (NULL) 0,	/* ob_size */
  "extractor.KeywordList",	/* tp_name */
  sizeof (KeywordList),		/* tp_basicsize */
  0,				/* tp_itemsize */
  (destructor) KeywordList_dealloc,	/* tp_dealloc */
  0,				/* tp_print */
  0,				/* tp_getattr */
  0,				/* tp_setattr */
  0,				/* tp_compare */
  0,				/* tp_repr */
  0,				/* tp_as_number */
  0,				/* tp_as_sequence */
  0,				/* tp_as_mapping */
  0,				/* tp_hash */
  0,				/* tp_call */
  (reprfunc) KeywordList_str,	/* tp_str */
  0,				/* tp_getattro */
  0,				/* tp_setattro */
  0,				/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,	/* tp_flags */
  "Keyword List Objects",	/* tp_doc */
  0,				/* tp_traverse */
  0,				/* tp_clear */
  0,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  (getiterfunc) KeywordList_iter,	/* tp_iter */
  0,				/* tp_iternext */
  KeywordList_methods,		/* tp_methods */
  0,				/* tp_members */
  0,				/* tp_getset */
  0,				/* tp_base */
  0,				/* tp_dict */
  0,				/* tp_descr_get */
  0,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  (initproc) KeywordList_init,	/* tp_init */
  0,				/* tp_alloc */
  KeywordList_new,		/* tp_new */
};

static PyObject *
KeywordListIter_new (PyTypeObject * type, PyObject * args, PyObject * kwargs)
{
  KeywordListIter *self;

  self = (KeywordListIter *) type->tp_alloc (type, 0);
  if (self != NULL)
    {
      self->list = NULL;
      self->pos = 0;
    }
  return (PyObject *) self;
}

static int
KeywordListIter_init (KeywordListIter * self, PyObject * args,
		      PyObject * kwargs)
{
  if (self->list)
    {
      Py_DECREF (self->list);
      self->list = NULL;
    }
  self->pos = 0;
  return 0;
}

static PyObject *
KeywordListIter_iter (KeywordListIter * self)
{
  Py_INCREF (self);
  return (PyObject *) self;
}

static PyObject *
KeywordListIter_iternext (KeywordListIter * self)
{
  PyObject *ret;
  EXTRACTOR_KeywordList *list;
  int pos;

  if (!(self->list))
    return NULL;
  list = self->list->keywords;
  pos = self->pos;
  while ((pos-- > 0) && list)
    list = list->next;
  if (list)
    {
      (self->pos)++;
      ret = PyTuple_New (2);
      PyTuple_SetItem (ret, 0, PyInt_FromLong (list->keywordType));
      PyTuple_SetItem (ret, 1, PyString_FromString (list->keyword));
      return ret;
    }
  return NULL;
}

static void
KeywordListIter_dealloc (KeywordListIter * self)
{
  if (self->list)
    {
      Py_DECREF (self->list);
    }
  self->ob_type->tp_free ((PyObject *) self);
}

static PyTypeObject KeywordListIterType = {
  PyObject_HEAD_INIT (NULL) 0,	/* ob_size */
  "extractor.KeywordList.KeywordListIter",	/* tp_name */
  sizeof (KeywordListIter),	/* tp_basicsize */
  0,				/* tp_itemsize */
  (destructor) KeywordListIter_dealloc,	/* tp_dealloc */
  0,				/* tp_print */
  0,				/* tp_getattr */
  0,				/* tp_setattr */
  0,				/* tp_compare */
  0,				/* tp_repr */
  0,				/* tp_as_number */
  0,				/* tp_as_sequence */
  0,				/* tp_as_mapping */
  0,				/* tp_hash */
  0,				/* tp_call */
  0,				/* tp_str */
  0,				/* tp_getattro */
  0,				/* tp_setattro */
  0,				/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT,		/* tp_flags */
  "Keyword List Iterator",	/* tp_doc */
  0,				/* tp_traverse */
  0,				/* tp_clear */
  0,				/* tp_richcompare */
  0,				/* tp_weaklistoffset */
  (getiterfunc) KeywordListIter_iter,	/* tp_iter */
  (iternextfunc) KeywordListIter_iternext,	/* tp_iternext */
  0,				/* tp_methods */
  0,				/* tp_members */
  0,				/* tp_getset */
  0,				/* tp_base */
  0,				/* tp_dict */
  0,				/* tp_descr_get */
  0,				/* tp_descr_set */
  0,				/* tp_dictoffset */
  (initproc) KeywordListIter_init,	/* tp_init */
  0,				/* tp_alloc */
  KeywordListIter_new,		/* tp_new */
};

/* Module level. */
static PyObject *
Extractor_defaultextractors (PyObject * self)
{
  PyObject *ret, *p;
  char buf[1024];
  char *b, *e;

  ret = PyList_New (0);
  b = (char *) EXTRACTOR_DEFAULT_LIBRARIES;
  while (*b != 0)
    {
      e = buf;
      while (*b != 0 && *b != ':')
	*(e++) = *(b++);
      *e = 0;
      p = PyString_FromString (buf);
      PyList_Append (ret, p);
      Py_DECREF (p);
      if (*b == ':')
	b++;
    }
  return ret;
}

static PyObject *
Extractor_encodebin (PyObject * self, PyObject * arg)
{
  PyObject *ret;

  if (PyString_Check (arg))
    return PyString_FromString (EXTRACTOR_binaryEncode ((unsigned char *)
							PyString_AsString
							(arg),
							PyString_Size (arg)));
  ret = Py_None;
  Py_INCREF (ret);
  return ret;
}

static PyObject *
Extractor_decodebin (PyObject * self, PyObject * arg)
{
  PyObject *ret;
  unsigned char *b;
  size_t l;

  ret = Py_None;
  if (!PyString_Check (arg))
    {
      Py_INCREF (ret);
    }
  else if (EXTRACTOR_binaryDecode (PyString_AsString (arg), &b, &l))
    {
      Py_INCREF (ret);
    }
  else
    ret = PyString_FromStringAndSize ((char *) b, l);
  return ret;
}

static PyMethodDef Extractor_Module_methods[] = {
  {"defaultExtractors", (PyCFunction) Extractor_defaultextractors,
   METH_NOARGS},
  {"encodeBinary", (PyCFunction) Extractor_encodebin, METH_O},
  {"decodeBinary", (PyCFunction) Extractor_decodebin, METH_O},
  {NULL}			/* Sentinel */
};

PyMODINIT_FUNC
initextractor (void)
{
  PyObject *m, *d;
  int i, n;

  if (PyType_Ready (&ExtractorListType))
    return;
  if (PyType_Ready (&KeywordListType))
    return;
  if (PyType_Ready (&KeywordListIterType))
    return;
  if ((m = Py_InitModule3 ("extractor", Extractor_Module_methods,
			   "Extractor module.")) == NULL)
    return;
  Py_INCREF (&ExtractorListType);
  PyModule_AddObject (m, "ExtractorList", (PyObject *) & ExtractorListType);
  Py_INCREF (&KeywordListType);
  PyModule_AddObject (m, "KeywordList", (PyObject *) & KeywordListType);
  d = PyDict_New ();
  n = EXTRACTOR_getHighestKeywordTypeNumber ();
  for (i = 0; i < n; i++)
    PyDict_SetItemString (d, (char *)
			  EXTRACTOR_getKeywordTypeAsString (i),
			  PyInt_FromLong (i));
  PyModule_AddObject (m, "KEYWORDTYPE", PyDictProxy_New (d));
  PyModule_AddIntConstant (m, "DUPLICATES_Typeless",
			   EXTRACTOR_DUPLICATES_TYPELESS);
  PyModule_AddIntConstant (m, "DUPLICATES_Remove_Unknown",
			   EXTRACTOR_DUPLICATES_REMOVE_UNKNOWN);
}
